<?php

namespace App\Http\Controllers;

use App\FileEntry;
use App\Http\Requests\CrupdateShareableLinkRequest;
use App\Services\Links\CrupdateShareableLink;
use App\Services\Links\GetShareableLink;
use App\ShareableLink;
use Common\Core\BaseController;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class ShareableLinksController extends BaseController
{
    /**
     * @var Request
     */
    private $request;

    /**
     * @var ShareableLink
     */
    private $link;

    /**
     * @var FileEntry
     */
    private $entry;

    public function __construct(Request $request, ShareableLink $link, FileEntry $entry)
    {
        $this->request = $request;
        $this->link = $link;
        $this->entry = $entry;
    }

    /**
     * @param int|string $idOrHash
     * @return JsonResponse
     */
    public function show($idOrHash)
    {
        $response = app(GetShareableLink::class)
            ->execute($idOrHash, $this->request->all());

        $link = $response['link'];
        if ( ! $link || ! $link->entry || $link->entry->trashed()) abort(404);

        $this->authorize('show', $link);

        return $this->success($response);
    }

    public function store(int $entryId, CrupdateShareableLinkRequest $request, CrupdateShareableLink $action)
    {
        $this->authorize('create', ShareableLink::class);
        $this->authorize('update', [FileEntry::class, [$entryId]]);

        $params = $request->all();
        $params['userId'] = $request->user()->id;
        $params['entryId'] = $entryId;

        $link = $action->execute($params);

        return $this->success(['link' => $link]);
    }

    /**
     * @param int $id
     * @param CrupdateShareableLinkRequest $request
     * @param CrupdateShareableLink $action
     *
     * @return JsonResponse
     */
    public function update($id, CrupdateShareableLinkRequest $request, CrupdateShareableLink $action)
    {
        $link = $this->link->findOrFail($id);

        $this->authorize('update', $link);

        $params = $request->all();
        $params['userId'] = $request->user()->id;

        $action->execute($params, $link);

        return $this->success(['link' => $link]);
    }

    /**
     * @param int $id
     * @return JsonResponse
     */
    public function destroy($id)
    {
        /**
         * @var ShareableLink $link
         */
        $link = $this->link->findOrFail($id);

        $this->authorize('destroy', $link);

        $link->delete();

        return $this->success();
    }
}
